# coding=utf8

from auto_keyworder_modules import easydb_api, eas
from auto_keyworder_modules.ai_services import ai_service_configuration

AUTO_KEYWORDER_ERROR = 'AUTO_KEYWORDER_ERROR'
AUTO_KEYWORDER_OBJECT_UPDATE = 'AUTO_KEYWORDER_OBJECT_UPDATE'
AUTO_KEYWORDER_OBJECT_CREATE = 'AUTO_KEYWORDER_OBJECT_CREATE'
AUTO_KEYWORDER_REQUEST_CLOUDSIGHT = 'AUTO_KEYWORDER_REQUEST_CLOUDSIGHT'
AUTO_KEYWORDER_REQUEST_DEEPVA = 'AUTO_KEYWORDER_REQUEST_DEEPVA'
AUTO_KEYWORDER_REQUEST_IMAGGA = 'AUTO_KEYWORDER_REQUEST_IMAGGA'


def log_event(
    easydb_server: str,
    easydb_token: str,
    event_data: dict = None,
):
    easydb_api.easydb_post(
        server=easydb_server,
        endpoint="event",
        easydb_token=easydb_token,
        payload=event_data,
    )


def log_object_events(
    easydb_server: str,
    easydb_token: str,
    event_infos: list,
):
    for info_json in event_infos:
        log_event(easydb_server, easydb_token, info_json)


# --------------------------------------------------


def format_ai_service_failed_event(
    service_config: ai_service_configuration.AiServiceConfiguration,
    error_message: str,
    asset_status: eas.AssetStatus,
    asset_version: str,
) -> dict:
    return {
        'type': AUTO_KEYWORDER_ERROR,
        'objecttype': asset_status.easydb_obj_objecttype(),
        'object_id': asset_status.easydb_obj_id(),
        'object_version': asset_status.easydb_obj_version(),
        'info': {
            'ai_service': {
                'url': service_config.get_api_url(),
                'job': asset_status.ai_job_id,
            },
            'error': error_message,
            'asset': {
                'eas_id': asset_status.eas_id,
                'extension': asset_status.img_type,
                'original_filepath': asset_status.original_filepath,
                'url': asset_status.img_url,
                'version': asset_version,
            },
        },
    }


def format_object_change_event(
    service_config: ai_service_configuration.AiServiceConfiguration,
    asset_status: eas.AssetStatus,
    asset_version: str,
) -> dict:
    version = asset_status.easydb_obj_version()
    ev = {
        'objecttype': asset_status.easydb_obj_objecttype(),
        'object_id': asset_status.easydb_obj_id(),
        'object_version': version,
        'info': {
            'ai_service': {
                'url': service_config.get_api_url(),
                'job': asset_status.ai_job_id,
                'parsed_data': asset_status.ai_data,
            },
            'language': service_config.get_api_language(),
            'asset': {
                'eas_id': asset_status.eas_id,
                'extension': asset_status.img_type,
                'original_filepath': asset_status.original_filepath,
                'url': asset_status.img_url,
                'version': asset_version,
            },
        },
    }

    typ = AUTO_KEYWORDER_OBJECT_CREATE
    if version > 1:
        typ = AUTO_KEYWORDER_OBJECT_UPDATE
        if len(asset_status.diff) > 0:
            ev['info']['changes'] = sorted(list(asset_status.diff))
        else:
            ev['info']['changes'] = False

    ev['type'] = typ

    return ev


def format_request_event(
    event_type: str,
    asset_status: eas.AssetStatus,
    request_method: str,
    request_url: str,
    request_body: dict,
    response_status_code: int,
    response_content: str,
) -> dict:
    return {
        'type': event_type,
        'objecttype': asset_status.easydb_obj_objecttype(),
        'object_id': asset_status.easydb_obj_id(),
        'object_version': asset_status.easydb_obj_version(),
        'info': {
            'asset': {
                'eas_id': asset_status.eas_id,
                'extension': asset_status.img_type,
                'original_filepath': asset_status.original_filepath,
                'url': asset_status.img_url,
            },
            'request': {
                'method': request_method,
                'url': request_url,
                'body': request_body,
            },
            'response': {
                'statuscode': response_status_code,
                'body': response_content,
            },
        },
    }
