# coding=utf8

# import sys
# import traceback
# import json

from auto_keyworder_modules import util


class FieldInfo(object):

    def __init__(
        self,
        typ: str,
        name: str,
        column_id: int,
        nested: str = None,
        link_name: str = None,
        linked_ot: str = None,
        linked_ot_mask: str = None,
    ):
        self.typ = typ
        self.name = name
        self.nested = nested
        self.column_id = column_id
        self.link_name = link_name
        self.linked_ot = linked_ot
        self.linked_ot_mask = linked_ot_mask

    def to_dict(self):
        d = {'type': self.typ, 'name': self.name}
        if self.nested is not None:
            d['nested'] = self.nested
        if self.link_name is not None:
            d['link_name'] = self.link_name
        if self.linked_ot is not None:
            d['linked_objecttype'] = self.linked_ot
        if self.linked_ot_mask is not None:
            d['linked_objecttype_mask'] = self.linked_ot_mask
        return d

    def equals(self, other) -> bool:
        if other is None:
            return False
        return self.column_id == other.column_id

    def is_in_nested(self) -> bool:
        return self.nested is not None

    def is_l10n(self) -> bool:
        return self.typ in ['text_l10n', 'text_l10n_oneline']

    def is_linked(self) -> bool:
        return (
            self.link_name is not None
            and self.linked_ot is not None
            and self.linked_ot_mask is not None
        )


def get_tables(datamodel: dict) -> list:
    return util.get_json_value(datamodel, 'schema.tables', True)


def get_masks(datamodel: dict) -> list:
    return util.get_json_value(datamodel, 'maskset.masks', True)


def get_table_for_objecttype(
    tables: list,
    objecttype: str,
) -> dict:
    for t in tables:
        name = util.get_json_value(t, 'name', True)
        if name == objecttype:
            # check if not nested table
            owned_by = util.get_json_value(t, 'owned_by')
            if owned_by is None:
                return t
            return None
    return None


def get_table_by_id(
    tables: list,
    table_id: int,
) -> dict:
    for t in tables:
        _id = util.get_json_value(t, 'table_id', True)
        if _id == table_id:
            # check if not nested table
            owned_by = util.get_json_value(t, 'owned_by')
            if owned_by is None:
                return t
            return None
    return None


def get_preferred_mask_for_objecttype(
    datamodel: dict,
    objecttype: str,
) -> str:
    masks = get_masks(datamodel)
    if not isinstance(masks, list):
        raise Exception('maskset is invalid (no masks found)')

    for mask in masks:
        table_name_hint = util.get_json_value(mask, 'table_name_hint')
        if table_name_hint != objecttype:
            continue

        is_preferred = util.get_json_value(mask, 'is_preferred')
        if not isinstance(is_preferred, bool) or not is_preferred:
            continue

        return util.get_json_value(mask, 'name')

    return None


def get_nested_table(
    tables: list,
    nested_name: str,
    owning_table: str = None,
) -> dict:
    for t in tables:
        name = util.get_json_value(t, 'name', True)
        if name == nested_name:
            # check if nested table
            other_table = util.get_json_value(t, 'owned_by.other_table_name_hint')
            if owning_table is None:
                if other_table is not None:
                    return t
            elif other_table == owning_table:
                return t
            return None
    return None


def get_field_from_table(
    table: dict,
    fieldname: str,
    fieldtypes: list,
) -> FieldInfo:
    fields = util.get_json_value(table, 'columns', True)
    if not isinstance(fields, list):
        raise Exception('datamodel is invalid (no columns in table found)')

    for field in fields:
        kind = util.get_json_value(field, 'kind')
        if kind != 'column':
            continue

        name = util.get_json_value(field, 'name')
        if name == fieldname:
            _type = util.get_json_value(field, 'type')
            if _type in fieldtypes:
                return FieldInfo(_type, name, util.get_json_value(field, 'column_id'))

    raise Exception(
        # f'no field "{fieldname}" with type(s) {fieldtypes} found in table {table["name"]}'
        f'no field "{fieldname}" with type(s) {fieldtypes} found in table {table}'
    )


def get_objecttype_for_link(
    table: dict,
    tables: list,
    fieldname: str,
) -> str:
    tablename = util.get_json_value(table, 'name')
    if not tablename:
        raise Exception('datamodel is invalid (no name in table)')

    fields = util.get_json_value(table, 'columns')
    if not isinstance(fields, list):
        raise Exception('datamodel is invalid (no columns in table found)')

    foreign_keys = util.get_json_value(table, 'foreign_keys')
    if not isinstance(foreign_keys, list) or len(foreign_keys) < 1:
        raise Exception(
            f'datamodel is invalid (no foreign_keys in table {tablename} found)'
        )

    link_field = get_field_from_table(table, fieldname, ['link'])
    if link_field is None:
        raise Exception(f'link {fieldname} not found in {tablename}')
    if link_field.column_id is None:
        raise Exception(f'column id not found in link {fieldname} in {tablename}')

    link_table_id = None
    for f in foreign_keys:
        if not 'columns' in f:
            continue
        for c in f['columns']:
            if not 'column_id' in c:
                continue
            if c['column_id'] == link_field.column_id:
                if not 'referenced_table' in f:
                    continue
                ref = f['referenced_table']
                link_table_id = ref['table_id']
    if not isinstance(link_table_id, int):
        raise Exception('could not get linked table id in foreign keys')

    link_table = get_table_by_id(tables, link_table_id)
    return util.get_json_value(link_table, 'name')


def get_objecttype_for_link_in_nested(
    table: dict,
    tables: list,
    nested_name: str,
    fieldname: str,
) -> str:
    fields = util.get_json_value(table, 'columns')
    if not isinstance(fields, list):
        raise Exception('datamodel is invalid (no columns in table found)')

    # find the nested table in the current table
    nested_name_valid = False
    for field in fields:
        kind = util.get_json_value(field, 'kind')
        if kind != 'link':
            continue

        name = util.get_json_value(field, 'other_table_name_hint')
        if name == nested_name:
            nested_name_valid = True
            break
    if not nested_name_valid:
        raise Exception(f'nested table {nested_name} is not linked in table')

    nested_table = get_nested_table(tables, nested_name)

    foreign_keys = util.get_json_value(nested_table, 'foreign_keys')
    if not isinstance(foreign_keys, list) or len(foreign_keys) < 1:
        raise Exception(
            f'datamodel is invalid (no foreign_keys in nested table {nested_name} found)'
        )

    link_field = get_field_from_table(nested_table, fieldname, ['link'])
    if link_field is None:
        raise Exception(f'link {fieldname} not found in {nested_name}')
    if link_field.column_id is None:
        raise Exception(f'column id not found in link {fieldname} in {nested_name}')

    link_table_id = None
    for f in foreign_keys:
        if not 'columns' in f:
            continue
        for c in f['columns']:
            if not 'column_id' in c:
                continue
            if c['column_id'] == link_field.column_id:
                if not 'referenced_table' in f:
                    continue
                ref = f['referenced_table']
                link_table_id = ref['table_id']
    if not isinstance(link_table_id, int):
        raise Exception('could not get linked table id in foreign keys')

    link_table = get_table_by_id(tables, link_table_id)
    return util.get_json_value(link_table, 'name')


def get_field_info(
    datamodel: dict,
    objecttype: str,
    fieldname: str,
    fieldtypes: list,
    allow_in_nested: bool = False,
) -> FieldInfo:

    fieldparts = fieldname.split('.')
    if len(fieldparts) < 2 or len(fieldparts) > 4:
        raise Exception(f'field name "{fieldname}" is invalid')

    if not allow_in_nested and len(fieldparts) >= 3:
        raise Exception(f'field name "{fieldname}" is inside a nested which is invalid')

    if fieldparts[0] != objecttype:
        raise Exception(f'field "{fieldname}" is not in table "{objecttype}"')

    tables = get_tables(datamodel)
    if not isinstance(tables, list):
        raise Exception(f'datamodel is invalid (no tables found)')

    table = get_table_for_objecttype(tables, objecttype)
    if table is None:
        raise Exception(f'objecttype "{objecttype}" not found in datamodel')

    if len(fieldparts) == 2:
        # find field in top_level
        info = get_field_from_table(table, fieldparts[1], fieldtypes)
        if info is not None:
            return info
    else:
        nested_table = None
        _nested_name = None
        if fieldparts[1].startswith('_nested:'):
            _nested_name = fieldparts[1][len('_nested:') :]
            nested_table = get_nested_table(tables, _nested_name, objecttype)

        if len(fieldparts) == 3:
            if nested_table is None:
                # find field in linked object in top level
                linked_ot = get_objecttype_for_link(table, tables, fieldparts[1])
                if linked_ot is not None:
                    linked_table = get_table_for_objecttype(tables, linked_ot)
                    info = get_field_from_table(linked_table, fieldparts[2], fieldtypes)
                    if info is not None:
                        info.link_name = fieldparts[1]
                        info.linked_ot = linked_ot
                        info.linked_ot_mask = get_preferred_mask_for_objecttype(
                            datamodel, linked_ot
                        )
                        return info
            else:
                # find field in nested table
                info = get_field_from_table(nested_table, fieldparts[2], fieldtypes)
                if info is not None:
                    info.nested = fieldparts[1]
                    return info
        else:
            # find field in linked object in nested table
            linked_ot = get_objecttype_for_link_in_nested(
                table, tables, _nested_name, fieldparts[2]
            )
            if linked_ot is not None:
                linked_table = get_table_for_objecttype(tables, linked_ot)
                info = get_field_from_table(linked_table, fieldparts[3], fieldtypes)
                if info is not None:
                    info.nested = fieldparts[1]
                    info.link_name = fieldparts[2]
                    info.linked_ot = linked_ot
                    info.linked_ot_mask = get_preferred_mask_for_objecttype(
                        datamodel, linked_ot
                    )
                    return info

    raise Exception(f'field "{fieldname}" is not in table "{objecttype}"')


def check_objecttype_valid(
    datamodel: dict,
    objecttype: str,
) -> None:
    tables = get_tables(datamodel)
    if not isinstance(tables, list):
        raise Exception('datamodel is invalid (no tables found)')
    if get_table_for_objecttype(tables, objecttype) is None:
        raise Exception(f'objecttype "{objecttype}" not found in datamodel')
